/*********************************************************************

SD Card Web Server, modified from Microchip's TCP/IP Stack demo application (stack ver 5.00)
by Mauro Grassi, for SC.

Compilation Instructions:

Large Code Model, Constants in Code Space.
Large Data Model, Small Scalar Model.
Omit Frame Pointer, Procedural Abstraction.
Pre and Post Instr Scheduling Enabled.
Don't create Default ISRs.
Smallest Code Space Optimization Enabled.

Version History:

1.10	: this version has SD card reading routines working (sharing the SPI bus with EEPROM (to be deleted) and ENC28J60)
1.20	: transitional (fixed intermittent problem with ethernet connectivity - to do with sharing the SPI bus)
1.24	: added new commands to FTP server (like RETR) and made it read/write from/to SD card instead of EEPROM.
1.26	: added new commands to FTP server (like PWD and CWD)
1.28	: added SD card support for the HTTP server.
1.30	: made all three SPI speeds the same (8Mhz).
1.32	: transitional.
1.40	: fixed PutFile() method in FTP.c (bug with last few characters of file dropped).
1.44	: moved the doUARTConfig out, some other minor changes.
1.46	: experimental.
1.48	: transitional.
1.50	: added fsm support for ADC conversions.
1.52	: fixed FTP server file transfers. The Put was sometimes cutting out the end bit of the file. Improved speed of FTP server GET and PUT.
1.54	: transitional
1.56	: fixed intermittent connection problems due ultimately to the SPI bus sharing.
1.60	: adding support for configuration file. If the file is found, the settings from the file are loaded, if not, settings from EEPROM are loaded. Any file settings are copied to EEPROM.
1.70	: added more support for settings... Increased Optimization Level.
1.72	: added support for setting the FTP password and user name in the settings file.
1.80	: added support for TELNET password (the same as the FTP password). Note however, that the file settings can be viewed on a browser and it will show the password and username.
1.82	: blocked access to the settings file from HTTP for security reasons.
1.84	: added support for creating a directory if it doesn't exist in the FTP server.
1.86	: added blinking LED1 to indicate disk (ie SD card) activity.
1.88	: transitional. 
1.90	: support for ls and dir for the FTP Server added. Improved FTP PUT and GET speeds.
1.92	: added support for relative pathnames for the FTP server cd command.
1.94	: added true support for long file names.
1.96	: added support for directories in the HTTP GET function. 
2.00    : first version on PC board rather than breadboard. Different pin outs.
2.02	: transitional. Fixed EEPROM Page size to 16.
2.10	: experimental. Added dynamic variable support for firmware version. FIRMWARE_VERSION is declared in maindemo.h. 
		  improved the ftoa function.
2.12	: removed some modules and added Dynamic DNS support. 
2.13	: Also enabled DHCP settable in settings file.
2.14	: fixed a small bug with ProcessIO hanging separated the SAMP=0 & SAMP=1 states.
2.20	: works well.
2.22	: transitional.
2.30	: rewrote the ftoa function (it was buggy from j being an unsigned int for eg.). Made it so only - sign is shown.
2.32	: transitional.
2.40	: added support for time strings from NTP time (clock.c and clock.h). NTP Time is now a dynamic variable in the HTTP server.
2.42	: transitional.
2.50	: dynamic DNS works.
2.52	: removed Telnet server due to lack of program memory.
2.60	: improved float extraction and settings, etc.
2.70	: adding more settings including NTP Timezone etc. Added remote RESET capability through HTTP. More extensive logging...
2.72    : deleted some unused functions from FAT module.
2.74    : added week day support for NTP.
2.80    : added better epoch time string calculation for NTP. Valid for the whole 32 bit range of NTP epoch seconds from 1970-2106...
2.82	: transitional.
2.90	: streamlined variable settings file & creation of new settings file. Added values file. Added FTP Delete command.
2.92	: transitional.
2.94	: first case version - swapped LEDs around.
2.96	: cleaned up the SMTP client code - removed ROM pointers. Now with Email.
2.98	: added variable parse support for email message body.
2.99	: transitional
3.00	: changed Serial Port Speed to 115200 baud. Logging changes in the SMTP module... 
3.01    : fixed SMTP end of email send bug - kept trying to send email when failed.
3.10    : verified this version works with Optus SMTP outgoing mail server on port 25 (if unblocked). Fixed last character omitted bug in the emailParse function (parse for VARs).
3.12	: added logging for each variable.
3.14	: removed DHCP client - now we must use only fixed IP address.
3.16	: added log size with email log files
3.20	: major changes to the email sending, now can send an unlimited length email body.
3.22    : transitional.
3.24    : fixed some bugs with email sending...
3.26    : fixes some bugs with FTP.
3.30    : good version...
3.33    : fixes slow HTTP.
3.40    : fast FTP and HTTP...
3.50    : trasitional.
3.60    : fast FTP and HTTP and fixes problem with pdfs not being correctly put by putFile in the FTP server. The Tx FIFO must be twice the FILE_BUFFER_SIZE
3.70	: added default names for variables... Changed the variable indicator first character...
3.80	: fixed a bug with decp setting in variables not being a long. Also httpPort, DynDNSIPPort and DynDNSDynIPPort.
		  Also changed all settings to be saved to values file. Added restore defaults command. Added a few extra variables
		  for the three important file names used SETTINGS_FILE LOG_FILE and VALUES_FILE which are constants of compilation.
		  Also modified the .cgi handlers so that the last file name is restored on returning from executing the command. Allowing
		  different .cgi pages. Added last known public IP to variables. Prevented VALUES_FILE from being viewable/downloadable in a browser - security reasons.
		  Adding SD card information variables... Also fixed small bug in the detection of SD card for type SD v1 was being detected as MMC.
3.90	: Added Ping. Added remote Serial Port Write capability. Added last received Ping time and logged Ping replies including remote IP Address.
3.92	: transitional.
3.93    : experimental versions with tentative HTTP Authentication.
4.00    : HTTP Authentication added. Major change to dynamic variables. Added HTTP File Permissions and extensions. Rewrote to use only base64encode (to save space).
4.10    : added FTP Timeout variable.
4.20    : small fix to ICMP module to avoid errors when disabled at compile time.
4.30	: fixed small bug with variable logs. Renumbered the variable logs to start from 1.
4.40	: added (internal) code for generating dynamic variables legend file.
4.50	: normalized escape sequence for dynamic variable substitutions in dynamic pages. Now repeat the character to escape it, ie. use ~~ for ~ where ~ is the dynamic var indicator.
4.60	: rearranged default setting values for general release. Removed the EEPROM write routines. Added command for sending a file via email.
4.70	: added create settings command. Modified permission settings so they are also saved in the created settings file and there are different read only permission strings...
4.80	: added support for the optional Time Module connected via the serial port.
4.90	: added support for automatic baud rate for the Time Module.
4.92	: small improvements to the serial interface including to the Time Module.
4.94	: transitional
5.00	: First Release Version. 
5.10	: minor changes and saved some space by changing from default scalar model to small scalar model.
5.20	: fixed the file permissions not being saved and you can now set the daylight saving time offset. This also
		: allows you to turn it off by setting it to 0. Some places in Australia use half an hour offset for DST.
5.30	: SD card driver fix for 2Gb SD cards.
5.31	: transposed the four digital output pins to suit Jaycar's PC board overlay.
5.32	: transposed the analog inputs to suit Jaycar's PC board overlay.
5.33	: added a fix suggested by Richard Collett in an email (7 Jan 2010) to the FTP PWD command. (M.G.)
5.40	: fixed file permission problem where no password was being asked for private file extensions if using a different IP address from the default like 192.168.0.34 (eg 10.1.1.34). It turns out this bug was caused by the restData buffer
		: which was then on the stack (inside the HTTPProcess function) being overwritten with calls to HTTPParse. Declaring it static solves the problem (now on heap) (M.G.)
5.41    : added workaround for buggy SD card controllers
*/
     
#define THIS_IS_STACK_APPLICATION

// Include all headers for any enabled TCPIP Stack functions
#include "h/common.h"
#include "h/MainDemo.h"
#include "h/spi.h"
#include "h/sd.h"
#include "h/ff.h"
#include "h/TCPIP.h"
#include "h/clock.h"
#include "h/GenericTypeDefs.h"

// Declare AppConfig structure and some other supporting stack variables
// Global Variables
VARIABLE_TYPE	variable[NUM_VARS];
APP_CONFIG AppConfig;
BYTE tempString[16];
BYTE myDHCPBindCount=0xFF;
FATFS 	fileSystem;
FIL		fsrc;
BYTE	fileBuffer[FILE_BUFFER_SIZE];
int 	adcState;
int 	adcIndex;
char 	variableBuffer[80];
char 	valueBuffer[256];
char 	iBuffer[256];
char    valuesBuffer[256];
ROM 	char defaultUserName[] = "admin";
ROM 	char defaultPassWord[] = "pass";
ROM 	char defaultNTPServer[] = "pool.ntp.org";
ROM		char defaultDynDNSServer[] = "dynupdate.no-ip.com";					// NO_IP_COM
ROM		char defaultDynDNSUsername[] = "";
ROM		char defaultDynDNSPassword[] = "";
ROM		char defaultDynDNSHost[] = "";
ROM		char defaultDynDNSDynIPServer[] = "checkip.dyndns.com";
BYTE	defaultDHCPEnabled;
char 	userName[HIDDEN_STRING_MAX];
char 	passWord[HIDDEN_STRING_MAX];
FIL 		flog;
char		varlogFileName[32];
IP_ADDR		currentIPAddress;

extern int 	SMTPResolved;
extern TCP_SOCKET MySocket;
long    	baudRate;
BYTE 		procEmailNum;
BYTE 		sendEmailPending;
char 		sendEmailFile[SHORT_STRING_MAX];
char 		sendEmailSubject[SHORT_STRING_MAX];

#if !defined(STACK_USE_DHCP_CLIENT)
	#define DHCPBindCount	(1)
#endif

// Use UART2 instead of UART1 for stdout (printf functions).  Explorer 16 
// serial port hardware is on PIC UART2 module.
#if defined(EXPLORER_16)
	int __C30_UART = 2;
#endif

// Private helper functions.
// These may or may not be present in all applications.
void getLogEmailFileName(char*, int);
static void InitAppConfig(void);
static void InitializeBoard(void);
static void ProcessIO(void);
static void extractAllSettings(void);
float extractFloat(char*);
extern char* strcpyram2ram(BYTE*, BYTE*);
void DisplayIPValue(char*, IP_ADDR);

#if defined(STACK_USE_ICMP_SERVER)
extern DWORD lastPing;
extern IP_ADDR lastPingIP;
#endif

extern long icmpEnable;
extern DWORD lastFTP;
extern IP_ADDR lastFTPIP;
extern long lastFTPResult;
void initFileTypes(void);
extern BYTE httpFiles[TOTAL_FILE_TYPES][SHORT_STRING_MAX];
extern long httpFilesPermissions[TOTAL_FILE_TYPES];
extern BYTE httpContents[TOTAL_FILE_TYPES][SHORT_STRING_MAX];
extern BYTE lastFileType;
extern long FTPTimeout;
long timeModule;
long timeSMonth;
long timeSWeek;
long timeSDay;
long timeEMonth;
long timeEWeek;
long timeEDay;
long daylightSavingOffset;

SETTING settingsNames[TOTAL_SETTINGS]
=
{
// the following are constants set in the settings file...
 	{ (long*)0, "IPAddress", 				(BYTE*)&AppConfig.MyIPAddr.Val, 			RESULT_IP_ADDR			},	// 0	
	{ (long*)0, "DefaultIPAddress",		    (BYTE*)&AppConfig.DefaultIPAddr.Val, 		RESULT_IP_ADDR			},
	{ (long*)0, "SubnetMask",				(BYTE*)&AppConfig.MyMask.Val,				RESULT_IP_ADDR			},
	{ (long*)0, "DefaultMask",				(BYTE*)&AppConfig.DefaultMask.Val,			RESULT_IP_ADDR			},
	{ (long*)0, "Gateway",					(BYTE*)&AppConfig.MyGateway.Val,			RESULT_IP_ADDR			},
	{ (long*)0, "PrimaryDNS",				(BYTE*)&AppConfig.PrimaryDNSServer.Val,		RESULT_IP_ADDR			}, // 5
	{ (long*)0, "SecondaryDNS",			    (BYTE*)&AppConfig.SecondaryDNSServer.Val,	RESULT_IP_ADDR			},
	{ (long*)0, "Password",				    (BYTE*)&passWord,							RESULT_SHORT_STRING		},
	{ (long*)0, "User",					    (BYTE*)&userName,							RESULT_SHORT_STRING		},
	{ (long*)0, "HTTPPort",				    (BYTE*)&httpPort,							RESULT_INT				},
	{ (long*)0, "NTPServer",				(BYTE*)&NTPServer,							RESULT_STRING			}, // 10
	{ (long*)0, "NTPTimezone",				(BYTE*)&NTPTimezone,						RESULT_INT				},
	{ (long*)0, "DynDNSServer",			    (BYTE*)&DynDNSServer,						RESULT_STRING			},
	{ (long*)0, "DynDNSServerPort",		    (BYTE*)&DynDNSServerPort,					RESULT_INT				},
	{ (long*)0, "DynDNSUsername",			(BYTE*)&DynDNSUsername,						RESULT_STRING			},
	{ (long*)0, "DynDNSPassword",			(BYTE*)&DynDNSPassword,						RESULT_STRING			}, // 15
	{ (long*)0, "DynDNSHost",				(BYTE*)&DynDNSHost,							RESULT_SHORT_STRING		},
	{ (long*)0, "DynDNSDynIPServer",		(BYTE*)&DynDNSDynIPServer,					RESULT_SHORT_STRING		},
	{ (long*)0, "DynDNSDynIPPort",			(BYTE*)&DynDNSDynIPPort,					RESULT_INT				},
	{ (long*)0, "DynDNSForceUpdate",		(BYTE*)&defaultbForceUpdate,				RESULT_BOOLEAN			},
	{ (long*)0, "EmailAddress",			    (BYTE*)&emailAddress,						RESULT_SHORT_STRING		}, // 20
	{ (long*)0, "EmailServer",				(BYTE*)&emailServer,						RESULT_SHORT_STRING		},
	{ (long*)0, "EmailFrom",				(BYTE*)&emailFrom,							RESULT_SHORT_STRING		},
	{ (long*)0, "EmailUser",				(BYTE*)&emailUser,							RESULT_STRING			},
	{ (long*)0, "EmailPassword",			(BYTE*)&emailPassword,						RESULT_STRING			},
 	{ (long*)&variable[0].decp, "var0g",	(BYTE*)&variable[0].gradient, 				RESULT_FLOAT			}, // 25
 	{ (long*)&variable[0].decp, "var0y",	(BYTE*)&variable[0].yintercept,				RESULT_FLOAT			},		
 	{ (long*)&variable[0].decp, "var0min",	(BYTE*)&variable[0].min, 					RESULT_FLOAT			},		
 	{ (long*)&variable[0].decp, "var0max",	(BYTE*)&variable[0].max, 					RESULT_FLOAT			},		
 	{ (long*)0, "var0decp", 				(BYTE*)&variable[0].decp, 					RESULT_INT				},
 	{ (long*)0, "var0name",				    (BYTE*)&variable[0].name, 					RESULT_STRING			},	// 30
	{ (long*)0, "var0logp",				    (BYTE*)&variable[0].logp,					RESULT_INT				},
	{ (long*)0, "var0logsize",				(BYTE*)&variable[0].logsize,				RESULT_INT				},
 	{ (long*)&variable[1].decp, "var1g",	(BYTE*)&variable[1].gradient, 				RESULT_FLOAT			},		
 	{ (long*)&variable[1].decp, "var1y",	(BYTE*)&variable[1].yintercept,				RESULT_FLOAT			},		
 	{ (long*)&variable[1].decp, "var1min",	(BYTE*)&variable[1].min, 					RESULT_FLOAT			}, // 35		
 	{ (long*)&variable[1].decp, "var1max",	(BYTE*)&variable[1].max, 					RESULT_FLOAT			},		
 	{ (long*)0, "var1decp",				    (BYTE*)&variable[1].decp, 					RESULT_INT				},
 	{ (long*)0, "var1name",     			(BYTE*)&variable[1].name, 					RESULT_STRING			},		
	{ (long*)0, "var1logp",		    		(BYTE*)&variable[1].logp,					RESULT_INT				},
	{ (long*)0, "var1logsize",				(BYTE*)&variable[1].logsize,				RESULT_INT				}, // 40
 	{ (long*)&variable[2].decp, "var2g",	(BYTE*)&variable[2].gradient, 				RESULT_FLOAT			},		
 	{ (long*)&variable[2].decp, "var2y",	(BYTE*)&variable[2].yintercept,				RESULT_FLOAT			},		
 	{ (long*)&variable[2].decp, "var2min",	(BYTE*)&variable[2].min, 					RESULT_FLOAT			},		
 	{ (long*)&variable[2].decp, "var2max",	(BYTE*)&variable[2].max, 					RESULT_FLOAT			},		
 	{ (long*)0, "var2decp",				    (BYTE*)&variable[2].decp, 					RESULT_INT				}, // 45
 	{ (long*)0, "var2name",			    	(BYTE*)&variable[2].name, 					RESULT_STRING			},		
	{ (long*)0, "var2logp",		    		(BYTE*)&variable[2].logp,					RESULT_INT				},
	{ (long*)0, "var2logsize",				(BYTE*)&variable[2].logsize,				RESULT_INT				},
 	{ (long*)&variable[3].decp, "var3g",	(BYTE*)&variable[3].gradient, 				RESULT_FLOAT			},		
 	{ (long*)&variable[3].decp, "var3y",	(BYTE*)&variable[3].yintercept,				RESULT_FLOAT			},	// 50	
 	{ (long*)&variable[3].decp, "var3min",	(BYTE*)&variable[3].min, 					RESULT_FLOAT			},		
 	{ (long*)&variable[3].decp, "var3max",	(BYTE*)&variable[3].max, 					RESULT_FLOAT			},		
 	{ (long*)0, "var3decp",				    (BYTE*)&variable[3].decp, 					RESULT_INT				},
 	{ (long*)0, "var3name",				    (BYTE*)&variable[3].name, 					RESULT_STRING			},		
	{ (long*)0, "var3logp",				    (BYTE*)&variable[3].logp,					RESULT_INT				}, // 55
	{ (long*)0, "var3logsize",				(BYTE*)&variable[3].logsize,				RESULT_INT				},
	{ (long*)0, "SMTPPort",				    (BYTE*)&smtpPort,           				RESULT_INT				},
	{ (long*)0, "FTPPort",				    (BYTE*)&ftpPort,             				RESULT_INT				},
	{ (long*)0, "FTPDataPort",				(BYTE*)&ftpDataPort,           				RESULT_INT				},
	{ (long*)0, "Baudrate", 				(BYTE*)&baudRate,           				RESULT_INT				}, // 60
	{ (long*)0, "NTPPort", 				    (BYTE*)&ntpPort,           				    RESULT_INT				}, 
	{ (long*)0, "ICMPEnable", 				(BYTE*)&icmpEnable,           				RESULT_INT				}, 
	{ (long*)0, "p0", 				    (BYTE*)&httpFilesPermissions[0],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p1", 				    (BYTE*)&httpFilesPermissions[1],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p2", 				    (BYTE*)&httpFilesPermissions[2],  			RESULT_PERMISSION				},  //65
	{ (long*)0, "p3", 				    (BYTE*)&httpFilesPermissions[3],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p4", 				    (BYTE*)&httpFilesPermissions[4],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p5", 				    (BYTE*)&httpFilesPermissions[5],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p6", 				    (BYTE*)&httpFilesPermissions[6],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p7", 				    (BYTE*)&httpFilesPermissions[7],  			RESULT_PERMISSION				}, // 70
	{ (long*)0, "p8", 				    (BYTE*)&httpFilesPermissions[8],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p9", 				    (BYTE*)&httpFilesPermissions[9],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p10", 				    (BYTE*)&httpFilesPermissions[10],  			RESULT_PERMISSION				}, 
	{ (long*)0, "p11", 				    (BYTE*)&httpFilesPermissions[11],  			RESULT_PERMISSION				}, 
	{ (long*)0, "f0",					(BYTE*)&httpFiles[0],				RESULT_SHORT_STRING		}, // 75
	{ (long*)0, "f1",					(BYTE*)&httpFiles[1],				RESULT_SHORT_STRING		},
	{ (long*)0, "f2",					(BYTE*)&httpFiles[2],				RESULT_SHORT_STRING		},
	{ (long*)0, "f3",					(BYTE*)&httpFiles[3],				RESULT_SHORT_STRING		},
	{ (long*)0, "f4",					(BYTE*)&httpFiles[4],				RESULT_SHORT_STRING		},
	{ (long*)0, "f5",					(BYTE*)&httpFiles[5],				RESULT_SHORT_STRING		}, // 80
	{ (long*)0, "f6",					(BYTE*)&httpFiles[6],				RESULT_SHORT_STRING		},
	{ (long*)0, "f7",					(BYTE*)&httpFiles[7],				RESULT_SHORT_STRING		},
	{ (long*)0, "f8",					(BYTE*)&httpFiles[8],				RESULT_SHORT_STRING		},
	{ (long*)0, "f9",					(BYTE*)&httpFiles[9],				RESULT_SHORT_STRING		},
	{ (long*)0, "f10",					(BYTE*)&httpFiles[10],				RESULT_SHORT_STRING		}, // 85
	{ (long*)0, "f11",					(BYTE*)&httpFiles[11],				RESULT_SHORT_STRING		},
	{ (long*)0, "c0",					(BYTE*)&httpContents[0],			RESULT_SHORT_STRING		},
	{ (long*)0, "c1",					(BYTE*)&httpContents[1],			RESULT_SHORT_STRING		},
	{ (long*)0, "c2",					(BYTE*)&httpContents[2],			RESULT_SHORT_STRING		},
	{ (long*)0, "c3",					(BYTE*)&httpContents[3],			RESULT_SHORT_STRING		}, // 90
	{ (long*)0, "c4",					(BYTE*)&httpContents[4],			RESULT_SHORT_STRING		},
	{ (long*)0, "c5",					(BYTE*)&httpContents[5],			RESULT_SHORT_STRING		},
	{ (long*)0, "c6",					(BYTE*)&httpContents[6],			RESULT_SHORT_STRING		},
	{ (long*)0, "c7",					(BYTE*)&httpContents[7],			RESULT_SHORT_STRING		},
	{ (long*)0, "c8",					(BYTE*)&httpContents[8],			RESULT_SHORT_STRING		}, // 95
	{ (long*)0, "c9",					(BYTE*)&httpContents[9],			RESULT_SHORT_STRING		},
	{ (long*)0, "c10",					(BYTE*)&httpContents[10],			RESULT_SHORT_STRING		},
	{ (long*)0, "c11",					(BYTE*)&httpContents[11],			RESULT_SHORT_STRING		},
	{ (long*)0, "c12",					(BYTE*)&httpContents[12],			RESULT_SHORT_STRING		},
	{ (long*)0, "p12", 				    (BYTE*)&httpFilesPermissions[12],  	RESULT_PERMISSION		}, // 100
	{ (long*)0, "FTPTimeout", 		    (BYTE*)&FTPTimeout,  	            RESULT_INT		        }, 
	{ (long*)0, "tmMod",				(BYTE*)&timeModule,					RESULT_INT				},
	{ (long*)0, "tmSM",					(BYTE*)&timeSMonth,					RESULT_INT				},
	{ (long*)0, "tmSW",					(BYTE*)&timeSWeek,					RESULT_INT				},
	{ (long*)0, "tmSD",					(BYTE*)&timeSDay,					RESULT_INT				},
	{ (long*)0, "tmEM",					(BYTE*)&timeEMonth,					RESULT_INT				},
	{ (long*)0, "tmEW",					(BYTE*)&timeEWeek,					RESULT_INT				},
	{ (long*)0, "tmED",					(BYTE*)&timeEDay,					RESULT_INT				},
	{ (long*)0, "DSTOffset",			(BYTE*)&daylightSavingOffset,		RESULT_INT				},
	{ (long*)0, 0,							0, 								RESULT_NONE 			}
};

//
// PIC18 Interrupt Service Routines
// 
// NOTE: Several PICs, including the PIC18F4620 revision A3 have a RETFIE FAST/MOVFF bug
// The interruptlow keyword is used to work around the bug when using C18
//

#if defined(__18CXX)
	#if defined(HI_TECH_C)
	void interrupt low_priority LowISR(void)
	#else
	#pragma interruptlow LowISR
	void LowISR(void)
	#endif
	{
	    TickUpdate();
	}
	
	#if defined(HI_TECH_C)
	void interrupt HighISR(void)
	#else
	#pragma interruptlow HighISR
	void HighISR(void)
	#endif
	{
	    #if defined(STACK_USE_UART2TCP_BRIDGE)
			UART2TCPBridgeISR();
		#endif

		#if defined(ZG_CS_TRIS)
			zgEintISR();
		#endif // ZG_CS_TRIS
	}

	#if !defined(HI_TECH_C)
	#pragma code lowVector=0x18
	void LowVector(void){_asm goto LowISR _endasm}
	#pragma code highVector=0x8
	void HighVector(void){_asm goto HighISR _endasm}
	#pragma code // Return to default code section
	#endif

// C30 and C32 Exception Handlers
// If your code gets here, you either tried to read or write
// a NULL pointer, or your application overflowed the stack
// by having too many local variables or parameters declared.

#elif defined(__C30__)
	void _ISR __attribute__((__no_auto_psv__)) _AddressError(void)
	{
	    Nop();
		Nop();
	}
	void _ISR __attribute__((__no_auto_psv__)) _StackError(void)
	{
	    Nop();
		Nop();
	}
	
#elif defined(__C32__)
	void _general_exception_handler(unsigned cause, unsigned status)
	{
		Nop();
		Nop();
	}
#endif

static BYTE disFix(BYTE val)
{
	val=val & 0x0F;
	if(val>9)val+=7;
	return 0x30+val;
}

BYTE* printADec(unsigned int value, BYTE* outstr)
{
	if(value>99)value=99;
	*outstr++=disFix(value/10);
	*outstr++=disFix(value%10);
	return outstr;
}

/*
static void disA(BYTE val)
{
#ifdef STACK_USE_UART

	putcUART('0');
	putcUART('x');
	putcUART(disFix(val>>4));
	putcUART(disFix(val));

#endif
}

static void disAX(BYTE val)
{
#ifdef STACK_USE_UART 
	putcUART(disFix(val>>4));
	putcUART(disFix(val));
#endif
}
*/

static char* printA(char* outstr, BYTE val)
{
	*outstr++=disFix(val>>4);
	*outstr++=disFix(val);
	return outstr;
}

char* printWord(char* outstr, unsigned long x)
{
	*outstr++='0';
	*outstr++='x';
	outstr=printA(outstr, x>>24);
	outstr=printA(outstr, x>>16);
	outstr=printA(outstr, x>>8);
	outstr=printA(outstr, x);
	*outstr='\0';
	return outstr;
}
/*
static void disWord(unsigned long x)
{
	disA(x>>24);
	disAX(x>>16);
	disAX(x>>8);
	disAX(x);
}
*/
static void initVariables(void)
{
	int i;
	for(i=0; i<NUM_VARS; i++)
	{
		variable[i].raw=0;
		variable[i].value=0;
		variable[i].gradient=1.0;
		variable[i].yintercept=0.0;
		variable[i].min=0.0;
		variable[i].max=1024.0;
		variable[i].decp=1;
		variable[i].name[0]='\0';
		variable[i].condition = CONDITION_NORMAL;
		variable[i].oldCondition = CONDITION_NORMAL;
		variable[i].lastlogp=0;
		variable[i].logp=0;						// logging turned off for the moment...
		variable[i].logsize=0;					// emailing log file off for the moment...
		variable[i].lognum=1;
		getLogEmailFileName(iBuffer, i);
		f_unlink(iBuffer);
		strcpypgm2ram(variable[i].name, "VAR0");
		variable[i].name[3]='0'+i;
	}
}

unsigned long myatol(char* instr)
{
	unsigned long l;
	char c;

	l=0;
	while(*instr!='\0')
	{
	 c=*instr++;
	 if((c>='0')&&(c<='9'))
	 {
		l*=10;
		l+=c-'0';
	 }
	}
	return l;
}
    
static char* getEmailFileName(char* fileNameOut, int varNum, CONDITION condition)
{
	*fileNameOut++='v';
	*fileNameOut++='a';
	*fileNameOut++='r';
	*fileNameOut++='0'+(3 & varNum);
	switch(condition)
	{
		case CONDITION_NORMAL:
			*fileNameOut++='o';
			*fileNameOut++='k';
			break;
		case CONDITION_MIN:
			*fileNameOut++='m';
			*fileNameOut++='i';
			*fileNameOut++='n';
			 break;
		case CONDITION_MAX:
			*fileNameOut++='m';
			*fileNameOut++='a';
			*fileNameOut++='x';
			 break;
	}
	*fileNameOut++='.';
	*fileNameOut++='t';
	*fileNameOut++='x';
	*fileNameOut++='t';
	*fileNameOut='\0';
	return fileNameOut;
}

/*
void showEEPROMMemory(void)
{
	int i;
	int j;

	j=0;
	XEEBeginRead(0);
	for(i=0; i<0x100; i++)
	{
	if((i & 7)==0)disA(i);
	putrsUART("...");
	disA(XEERead());
	putrsUART(".");	
	if((i & 7)==7)putrsUART("\r\n");
	}
	XEEEndRead();
	putrsUART("\r\n");
}
*/

/*
void showAppConfig(void)
{	
	int i;
	BYTE* p;

	putrsUART("AppConfig is:\r\n");
	p=(BYTE*)&AppConfig;
	for(i=0; i<sizeof(AppConfig); i++)
	{
	disA(*p++);
	putcUART('.');
	}
	putrsUART("\r\n");
}
*/

void ftoa(char* outstr, float f, int numdec)
{
	int i;
	int j;
	unsigned long k;

#if(DEBUG_STACK_HTTP)
putrsUART("F1");
#endif
	
	if(numdec>6)numdec=6; else if(numdec<0)numdec=0;
	if(f>=0)
	{ 
	
	} 
	else 
	{ 
	*outstr++='-'; 
	f=-f; 
	}
	j=numdec;
	while(j>0)
	{
	f*=10.0;
	j--;
	}
	f+=0.5;
	k=(unsigned long)f;
	if(k==0){ tempString[0]='0'; tempString[1]='\0'; } else ultoa(k, (BYTE*)tempString);
	j=strlen((char*)tempString);
	j-=numdec;
	i=0;
    if(j<1)
    {
    *outstr++='0';
    *outstr++='.';
    j++;
    while(j<1)
    {
        *outstr++='0';
        j++;
    }   
    j=-1; 
    }
             
	while(tempString[i]!='\0')
	{
	if(i==j)*outstr++='.';
	*outstr++=tempString[i++];
	}
	
	*outstr='\0';
#if(DEBUG_STACK_HTTP)
putrsUART("F5");
#endif
}

static FRESULT logStringFile(char* fileName, char* instring, int logTime)
{
	// add the following string to the log file...
	FRESULT 	res;
	unsigned 	int	u;

	res=f_open(&flog, fileName, FA_OPEN_ALWAYS | FA_WRITE);
	res|=f_lseek(&flog, flog.fsize);
	if((logTime)&&(NTPTimeActive))
	{
	getTimeString((BYTE*)iBuffer);
	u=strlen(iBuffer);
	iBuffer[u++]=' ';
	iBuffer[u++]='@';
	iBuffer[u++]=' ';
	iBuffer[u]='\0';
	//putrsUART(iBuffer); 
	res|=f_write(&flog, iBuffer, u, &u);
	}
	//putrsUART(instring);
	res|=f_write(&flog, instring, strlen(instring), &u);
	res|=f_close(&flog);
	return res;
}

FRESULT logString(char* instring, int logTime)
{
	return logStringFile(LOG_FILE, instring, logTime);
}

static int getConfigFileLine(char* variable, char* value, FIL* source, int maxchars)
{
	// get the next line of variable = value from the file source...
	int i, u;
	char c;
	int var, val;

	i=0;
	var=0;
	val=0;
	while(1)
	{
	f_read(source, &c, 1, (UINT*)&u);
	if((c=='\r')||(c=='\n')||(u==0))break;
	 if(c=='=')i=1; 
	 else
	 if(c!=' ')
	 {
	 if(i==0)
	 {
		if(var<maxchars){ *variable++=c; var++; }
	 }
	 else 
	 {
		if(val<maxchars){ *value++=c; val++; }
	 }
 	 }
	}
	*variable='\0';
	*value='\0';
	if(u==0)return 0; else return 1;
}

long extractNumber(char* instring)
{
	// returns a number from a string, including an IP address in X.X.X.X format.
	int i, f;
	char c;
	long result;
	int sign;

	i=0;
	f=0;
	iBuffer[0]='\0';
	result=0;
	sign=0;

	while((*instring)!='\0')
	{
	c=*instring;
	if((c>='0')&&(c<='9'))break;
	if((*instring)=='-')sign^=1;
	instring++;
	}

	while(*instring!='\0')
	{
		c=*instring++;
		if((c>='0')&&(c<='9'))
		{
		iBuffer[i++]=c;
		f=1;
		}
		else
		{
			iBuffer[i]='\0';
			if(f==1)
			{
			result=result<<8;
			result+=(myatol(iBuffer));
			}
			f=0;
			i=0;
			iBuffer[0]='\0';
		}
	}
	iBuffer[i]='\0';
	if(f==1)
	{
	result=result<<8;
	result+=(myatol(iBuffer));
	}
	if(sign)result=-result;
	return result;
}

float extractFloat(char* instring)
{
	// returns a number from a string, including an IP address in X.X.X.X format.
	int i, f;
	char c;
	float result, g;
	int sign;

	result=0;
	sign=0;

	while((*instring)!='\0')
	{
	c=*instring;
	if((c>='0')&&(c<='9'))break;
	if((*instring)=='-')sign^=1;
	instring++;
	}

	f=0;
	i=0;
	iBuffer[0]='\0';
	while(*instring!='\0')
	{
		c=*instring++;
		if((c>='0')&&(c<='9'))
		{
		iBuffer[i++]=c;
		}
		else
		{
			iBuffer[i]='\0';
			if(f==0)
			{
			result=(float)myatol(iBuffer);
			}
			else
			{
				g=(float)(myatol(iBuffer));
				while(i--)g/=10.0;
				result+=g;
			}
			f++;
			i=0;
			iBuffer[0]='\0';
		}
	}

	iBuffer[i]='\0';
	if(f==0)result=(float)(myatol(iBuffer));
	else
	{
	g=(float)(myatol(iBuffer));
	while(i--)g/=10.0;
	result+=g;
	}
	if(sign)result=-result;
	return result;
}

char* DisplayMACAddr(char* outstr, BYTE* buf)
{
	BYTE x;
	int i;
	
	for(i=0; i<6; i++)
	{
	x=*buf++;
	outstr=printA(outstr, x);
	if(i!=5)*outstr++=':';
	}
	*outstr='\0';
	return outstr;
}

#define WRITE_DIR 	0
#define READ_DIR	1


static int moveSetting(BYTE* dest, BYTE* source, RESULT_TYPE result, int direction)
{
	int n, m, max;
	n=-1;
	switch(result)
	{
		case	RESULT_NONE:
		default:
				break;

		case	RESULT_IP_ADDR:
		case 	RESULT_INT:
		case    RESULT_PERMISSION:
		case	RESULT_FLOAT:
				n=4;
				break;

		case	RESULT_HIDDEN:
				n=0;
				max=HIDDEN_STRING_MAX;
				break;

		case	RESULT_STRING:
				n=0;
				max=STRING_MAX;
				break;

		case	RESULT_BOOLEAN:
				n=1;
				break;

		case	RESULT_MAC_ADDR:
				n=6;
				break;

		case   	RESULT_SHORT_STRING:
				n=0;
				max=SHORT_STRING_MAX;
				break;
	}

	m=n;
	if(n>0)
	{
			while(n--)
			{
				if(direction==WRITE_DIR)
				{
					*dest++=*source++;
				} else
				{
					*source++=*dest++;
				}
			}
	}
	else
	if(n==0)
	{
			if(direction==WRITE_DIR)
			{
				while(n<max)
				{
					*dest++=*source;
					n++;
					if(*source=='\0')break;
					source++;
				}
				m=n;
			}
			else
			{
				while(n<max)
				{
					*source++=*dest;
					n++;
					if(*dest=='\0')break;
					dest++;
				}
				m=n;
			}
	}
	return m;
}

void printValueType(char* outstr, RESULT_TYPE result, BYTE* valuep, long* numdec)
{
	IP_ADDR xip;
	float valuef;
	long value;
	BYTE ibuf[6];
	int i;
	BYTE* outp;

	switch(result)
	{
	case RESULT_IP_ADDR:
			outp=(BYTE*)&xip.Val;
			for(i=0; i<4; i++)
			{
			*outp++=*valuep++;
			}		
			DisplayIPValue(outstr, xip);
			break;

	case RESULT_MAC_ADDR:
			for(i=0; i<6; i++)
			{
			ibuf[i]=*valuep++;
			}		
			DisplayMACAddr(outstr, (BYTE*)&ibuf[0]);
			break;

	case RESULT_HIDDEN:
			*outstr++='*';
			*outstr='\0';
			break;

	case RESULT_BOOLEAN:
			value=(DWORD)(*valuep);
			ultoa((unsigned long)(value & 0xFF), (BYTE*)outstr);
			break;
			
	case RESULT_INT:
	case RESULT_PERMISSION:
			outp=(BYTE*)&value;
			for(i=0; i<4; i++)
			{
			*outp++=*valuep++;
			}
			if(value<0){ *outstr++='-'; value=-value; }
			ultoa((unsigned long)value, (BYTE*)outstr);
			break;

	case RESULT_STRING:
	case RESULT_SHORT_STRING:
			strcpyram2ram((BYTE*)outstr, valuep);
			break;

	case RESULT_FLOAT:
			outp=(BYTE*)&valuef;
			for(i=0; i<4; i++)
			{
			*outp++=*valuep++;
			}
			ftoa(outstr, valuef, *numdec);
			break;

	case RESULT_NONE:
	default:
			*outstr='\0';
			break;
	}
}

float getValueType(char* outstr, char* instr, RESULT_TYPE result, BYTE* value, long* numdec)
{
	// extract the value into float of the type result
	// print the value into outstr and get from instr
	float x;
	long xd;
	IP_ADDR xip;
	BYTE* ip;
	BYTE* op;
	int n;

	ip=0;
	op=0;
	x=0;
	xip.Val=0;
	n=0;
	switch(result)
	{
	case RESULT_IP_ADDR:
			xip.Val=swapl(extractNumber(instr));
			ip=(BYTE*)&xip.Val;
			op=(BYTE*)value;
			n=4;
			break;

	case RESULT_INT:
	case RESULT_PERMISSION:
			xd=extractNumber(instr);
			ip=(BYTE*)&xd;
			op=(BYTE*)value;
			n=4;
			break;

	case RESULT_HIDDEN:
	case RESULT_STRING:
	case RESULT_SHORT_STRING:
			n=0;
			ip=(BYTE*)instr;
			op=(BYTE*)value;
			break;
			
	case RESULT_BOOLEAN:
			xd=extractNumber(instr);
			ip=(BYTE*)&xd;
			op=(BYTE*)value;
			n=4;
			break;

	case RESULT_FLOAT:
			x=extractFloat(instr);
			ip=(BYTE*)&x;
			op=(BYTE*)value;
			n=4;
			break;

	case RESULT_MAC_ADDR:
	case RESULT_NONE:
	default:
			*outstr='\0';
			break;
	}
	moveSetting(op, ip, result, WRITE_DIR);
	printValueType(outstr, result, value, numdec);
	return x;
}

/*
unsigned long littleEndian(unsigned long x)
{
	unsigned long r;
	r= 0x000000FF & (x>>24);
	r+=0x0000FF00 & (x>>8);
	r+=0x00FF0000 & (x<<8);
	r+=0xFF000000 & (x<<24);
	return r;
}
*/

static int extractSetting(char* variable, char* value, SETTING* settingPtr)
{
	RESULT_TYPE result;
	int i;
  
	result=RESULT_NONE;
	i=0;
	while(1)
	{
	if(settingPtr[i].resultType==RESULT_NONE)break;
	if(!stricmppgm2ram((BYTE*)variable, (BYTE*)settingPtr[i].name))break;
	i++;
	}

	if(settingPtr[i].resultType!=RESULT_NONE)
	{
		result=settingPtr[i].resultType;
		getValueType(iBuffer, value, result, (BYTE*)valuesBuffer, settingPtr[i].type);
		setValue(i, settingPtr, (BYTE*)valuesBuffer, 0);
	}
	else i=-1;
	return i;
}

static void printSetting(char* outstr, char* middle, int i, SETTING* settingPtr, BYTE* value)
{
	if(settingPtr[i].resultType!=RESULT_NONE)
	{
	outstr=strcpyram2ram((BYTE*)outstr, (BYTE*)settingPtr[i].name);
	outstr=strcpyram2ram((BYTE*)outstr, (BYTE*)middle);
	printValueType(outstr, settingPtr[i].resultType, value, settingPtr[i].type);
	}
}

static ROM BYTE SerializedMACAddress[6] = {MY_DEFAULT_MAC_BYTE1, MY_DEFAULT_MAC_BYTE2, MY_DEFAULT_MAC_BYTE3, MY_DEFAULT_MAC_BYTE4, MY_DEFAULT_MAC_BYTE5, MY_DEFAULT_MAC_BYTE6};

static void getConstants(void)
{
	int i;
	if((f_open(&fsrc, SETTINGS_FILE, FA_OPEN_EXISTING | FA_READ))==FR_OK)
	{
		while(1)
		{
		 if((getConfigFileLine(variableBuffer, valueBuffer, &fsrc, 80))==0)break;
		 i=extractSetting(variableBuffer, valueBuffer, settingsNames);
		}
//		XEEBeginWrite(0x0000);
//	    XEEWrite(0xFF);
//		XEEEndWrite();
	}
	f_close(&fsrc);
}

void getValues(void)
{
	int i, n;
	unsigned int u;

	if((f_open(&fsrc, VALUES_FILE, FA_OPEN_EXISTING | FA_READ))==FR_OK)
	{
		i=0;
		n=0;
		while(1)
		{
			if(settingsNames[i].resultType==RESULT_NONE)break;
			f_lseek(&fsrc, n);
			f_read(&fsrc, valuesBuffer, 255, &u);
			n+=(moveSetting(settingsNames[i].value, (BYTE*)valuesBuffer, settingsNames[i].resultType, WRITE_DIR));
			i++;
		}
	} 
	f_close(&fsrc);
}

int getValueIndex(char* variable, SETTING* settingPtr)
{
	int i;

	i=0;
	while(1)
	{
		if(settingPtr[i].resultType==RESULT_NONE)break;
		if(!strcmppgm2ram(variable, settingPtr[i].name))break;
		i++;
	}
	return i;
}

void setValue(int index, SETTING* settingPtr, BYTE* valueBuffer, int timestamp)
{
	logString("Set Value Attempt: ", timestamp);
	if((timestamp==0)||((settingPtr[index].resultType!=RESULT_HIDDEN)&&(HTTPUnlocked())))
	{
	moveSetting(settingPtr[index].value, valueBuffer, settingPtr[index].resultType, WRITE_DIR);
	printSetting(iBuffer, " set to ", index, settingPtr, settingPtr[index].value);
	logString(iBuffer, 0);
	logString(". Granted!\r\n", 0);
	}
	else
	{
	printSetting(iBuffer, " set to ", index, settingPtr, valueBuffer);
	logString(iBuffer, 0);
	logString(". Denied!\r\n", 0);
	}
}

void updateValues(void)
{
	UBRG = ((GetPeripheralClock()/baudRate)/16)-1;
}

void createSettingsFile(char* filen)
{
	int i, n;
	unsigned int u;
	char c;
	long l;

    updateValues();	
	if((f_open(&fsrc, filen, FA_OPEN_ALWAYS | FA_WRITE))==FR_OK)
	{
		i=0;
		n=0;
		while(1)
		{
			if(settingsNames[i].resultType==RESULT_NONE)break;
			f_write(&fsrc, settingsNames[i].name, strlen(settingsNames[i].name), &u);
			c='=';
			f_write(&fsrc, &c, 1, &u);
			l=6;
			printValueType(iBuffer, settingsNames[i].resultType, settingsNames[i].value, (long*)&l);
			f_write(&fsrc, iBuffer, strlen(iBuffer), &u);
			c='\r';
			f_write(&fsrc, &c, 1, &u);
			c='\n';
			f_write(&fsrc, &c, 1, &u);
			i++;
		}
	} 
	f_close(&fsrc);
}
    
void putValues(void)
{
	int i, n;
	unsigned int u;

    updateValues();	
	if((f_open(&fsrc, VALUES_FILE, FA_CREATE_ALWAYS | FA_WRITE))==FR_OK)
	{
		i=0;
		n=0;
		while(1)
		{
			if(settingsNames[i].resultType==RESULT_NONE)break;
			n=(moveSetting(settingsNames[i].value, (BYTE*)valuesBuffer, settingsNames[i].resultType, READ_DIR));
			f_write(&fsrc, valuesBuffer, n, &u);
			i++;
		}
	} 
	f_close(&fsrc);
}

static CONDITION updateCondition(int varNum, CONDITION lastCondition)
{
	switch(lastCondition)
	{
			default:
			case CONDITION_NORMAL:
				if(variable[varNum].value>variable[varNum].max)return CONDITION_MAX;
				if(variable[varNum].value<variable[varNum].min)return CONDITION_MIN;
				break;
			case CONDITION_MAX:
				if(variable[varNum].value<=(0.9*variable[varNum].max))return CONDITION_NORMAL;
				break;
			case CONDITION_MIN:
				if(variable[varNum].value>=(1.1*variable[varNum].min))return CONDITION_NORMAL;
				break;
	}
	return lastCondition;
}

static int noChangeCondition(int varNum)
{
	if(variable[varNum].condition==variable[varNum].oldCondition)return 1; else return 0;
}

static void extractAllSettings(void)
{
	// load defaults first...
	int i;

    initFileTypes();
    FTPTimeout=FTP_TIMEOUT;
	sendEmailPending=0;
	sendEmailFile[0]='\0';
	sendEmailSubject[0]='\0';
    emailPending=0;
	emailAddress[0]='\0';
//	emailMessageBody[0]='\0';
	emailServer[0]='\0';
	emailFrom[0]='\0';
	emailSubject[0]='\0';
	emailUser[0]='\0';
	emailPassword[0]='\0';
	memcpypgm2ram((void*)&AppConfig.MyMACAddr, (ROM void*)SerializedMACAddress, sizeof(AppConfig.MyMACAddr));
	AppConfig.MyIPAddr.Val = MY_DEFAULT_IP_ADDR_BYTE1 | MY_DEFAULT_IP_ADDR_BYTE2<<8ul | MY_DEFAULT_IP_ADDR_BYTE3<<16ul | MY_DEFAULT_IP_ADDR_BYTE4<<24ul;
	AppConfig.DefaultIPAddr.Val = AppConfig.MyIPAddr.Val;
	AppConfig.MyMask.Val = MY_DEFAULT_MASK_BYTE1 | MY_DEFAULT_MASK_BYTE2<<8ul | MY_DEFAULT_MASK_BYTE3<<16ul | MY_DEFAULT_MASK_BYTE4<<24ul;
	AppConfig.DefaultMask.Val = AppConfig.MyMask.Val;
	AppConfig.MyGateway.Val = MY_DEFAULT_GATE_BYTE1 | MY_DEFAULT_GATE_BYTE2<<8ul | MY_DEFAULT_GATE_BYTE3<<16ul | MY_DEFAULT_GATE_BYTE4<<24ul;
	AppConfig.PrimaryDNSServer.Val = MY_DEFAULT_PRIMARY_DNS_BYTE1 | MY_DEFAULT_PRIMARY_DNS_BYTE2<<8ul  | MY_DEFAULT_PRIMARY_DNS_BYTE3<<16ul  | MY_DEFAULT_PRIMARY_DNS_BYTE4<<24ul;
	AppConfig.SecondaryDNSServer.Val = MY_DEFAULT_SECONDARY_DNS_BYTE1 | MY_DEFAULT_SECONDARY_DNS_BYTE2<<8ul  | MY_DEFAULT_SECONDARY_DNS_BYTE3<<16ul  | MY_DEFAULT_SECONDARY_DNS_BYTE4<<24ul;
	icmpEnable=1;
	httpPort=HTTP_PORT;
	ntpPort=NTP_SERVER_PORT;
	smtpPort=SMTP_PORT;
	ftpPort=FTP_COMMAND_PORT;
	ftpDataPort=FTP_DATA_PORT;
	strcpypgm2ram(userName, defaultUserName);
	strcpypgm2ram(passWord, defaultPassWord);
	strcpypgm2ram(NTPServer, defaultNTPServer);
	NTPTimezone = 36000L;			// 10 hours is the default
	strcpypgm2ram(DynDNSServer, defaultDynDNSServer);
	DynDNSServerPort= 80;			// default
	strcpypgm2ram(DynDNSUsername, defaultDynDNSUsername);
	strcpypgm2ram(DynDNSPassword, defaultDynDNSPassword);
	strcpypgm2ram(DynDNSHost, defaultDynDNSHost);
	strcpypgm2ram(DynDNSDynIPServer, defaultDynDNSDynIPServer);
	DynDNSDynIPPort = 80; 			// default
	defaultbForceUpdate = 1;		// default : TRUE
	defaultDHCPEnabled = 0;			// turn off
	currentIPAddress.Val= 0;
	BUTTON0_TRIS=0;
	BUTTON1_TRIS=0;
	BUTTON2_TRIS=0;
	BUTTON3_TRIS=0;
	AppConfig.Flags.bIsDHCPEnabled = (BOOL)defaultDHCPEnabled;
	AppConfig.Flags.bInConfigMode  = (BOOL)defaultDHCPEnabled;
	XEEBeginRead(0x00FA);
	for(i=0; i<6; i++)
	{
    AppConfig.MyMACAddr.v[i]=XEERead();
	}
	XEEEndRead();
	logString("System: Loaded MAC Address: ", 0);
	DisplayMACAddr(iBuffer, &AppConfig.MyMACAddr.v[0]);

	//putrsUART("MAC Address: ");
	//putrsUART(iBuffer);
	//putrsUART("\r\n");

	logString(iBuffer, 0);
	logString("\r\n", 0);
	//logString("System: Loading Settings...\r\n", 0);
	getConstants();
	getValues();
	putValues();
}

void getLogEmailFileName(char* outp, int i)
{
	strcpypgm2ram(outp, VAR0_LOG_FILE);
	outp[3]='0'+i;
}

static void ProcessEmails(void)
{
	int i;

	if(procEmailNum>=9)procEmailNum=0;
	if(emailPending==2)
	{	
		// finished sending an email... so reset state machine
		procEmailNum++;
		emailPending=0;		
	}
	else
	if(emailPending==1)
	{
		// do nothing as an email is in the pipeline...
	}
	else
	{
		if(procEmailNum<4)
		{
		 if(!noChangeCondition(procEmailNum))
		 {	
			variable[procEmailNum].oldCondition=variable[procEmailNum].condition;
			getEmailFileName((char*)emailFile, procEmailNum, variable[procEmailNum].condition);
			strcpypgm2ram((char*)emailSubject, "Variable Changed!");
			emailPending=1;
			
		 } else procEmailNum++;
		} else
		if(procEmailNum<8)
		{
			i=procEmailNum-4;
			if(variable[i].logsize>0)
			{
				if(variable[i].lognum>variable[i].logsize)
				{
					variable[i].lognum=1;
					getLogEmailFileName((char*)emailFile, i);
					strcpypgm2ram((char*)emailSubject, "Variable Log");
					emailPending=1;
				} else procEmailNum++;
			} 
			else procEmailNum++;
		}
		else
		if(procEmailNum==8)
		{
			if(sendEmailPending)
			{
				strcpypgm2ram((char*)emailFile, (char*)sendEmailFile);
				strcpypgm2ram((char*)emailSubject, (char*)sendEmailSubject);
				emailPending=1;
				sendEmailPending=0;
			} else procEmailNum++;
		}
	}
}
/*
void createDynamicVariablesFile(void)
{
	int i;
	unsigned int u;
	char c;

	f_open(&fsrc, DYNAMIC_VARS_FILE, FA_CREATE_ALWAYS | FA_WRITE);
	for(i=0; i<TOTAL_SETTINGS;i++)
	{
		f_write(&fsrc, settingsNames[i].name, strlen(settingsNames[i].name), &u);
		c='\t';
		f_write(&fsrc, &c, 1, &u);
		c='\t';
		f_write(&fsrc, &c, 1, &u);
		c='\t';
		f_write(&fsrc, &c, 1, &u);
		c=disFix(i>>4);
		f_write(&fsrc, &c, 1, &u);
		c=disFix(i);
		f_write(&fsrc, &c, 1, &u);
		c='\r';
		f_write(&fsrc, &c, 1, &u);
		c='\n';
		f_write(&fsrc, &c, 1, &u);
	}
	f_close(&fsrc);
}
*/


#if defined(__18CXX)
void main(void)
#else
int main(void)
#endif
{
	static TICK t0;
    static TICK t1;
	static TICK t2; 
	DWORD time, oldTime;
	TIME_T destTime;
	int timeIndex;
	int timeNum = 0;
	BYTE* timePtr = (BYTE*)0;
	int i;
	int emailSendLog, emailReset;
	BYTE* outp;

	t0=t1=t2=0;
	MySocket = INVALID_SOCKET;
    InitializeBoard();
	// Initialize stack-related hardware components that may be 
	// required by the UART configuration routines
    TickInit();
	f_mount(0, &fileSystem);
	if(InitSDMMCCardSPI(&cardInfo))
	{
		//putrsUART("SD Card Status: Ok.\r\n");
	} 
	else 
	{
		//putrsUART("Bad SD Card. \r\n");
	}	


/*
   long x;
    TIME_T timet;
    TIME_T otimet;
    int equal;
   
    putrsUART("Begin\r\n");
    
    x=0;
    while(1)
    {
    convertEpoch2TimeOld(x, &otimet);
    convertEpoch2Time(x, &timet);
    equal=1;
    if(timet.year!=otimet.year)equal=2;
    if(timet.month!=otimet.month)equal=3;
    if(timet.day!=otimet.day)equal=4;
    if(timet.wday!=otimet.wday)equal=5;
    if(timet.hours!=otimet.hours)equal=6;
    if(timet.mins!=otimet.mins)equal=7;
    if(timet.secs!=otimet.secs)equal=8;
    if(equal!=1)
        {
            putrsUART("Unequal at x: ");
            disWord(x);
            putrsUART(" at: ");
            disA(equal);
            putrsUART(" D: ");
            disA(timet.day);
            putrsUART("<>");
            disA(otimet.day);
            putrsUART(" M:");
            disA(timet.month);
            putrsUART("<>");
            disA(otimet.month);
            putrsUART(" Y:");
            disWord(timet.year);
            putrsUART("\r\n");
              DelayMs(2000);
        }      
       else
       {
           if((x & 0xFFFF)==0)
           {
           putrsUART("Equal up to x: ");
           disWord(x);
           putrsUART(" Year: ");
           disWord(timet.year);
           putrsUART("\r\n");
           }     
       }
       x+=86400;    
    }
 */
      
	// erase what log there was...
	daylightSavingOffset=3600;		// one hour
	time=0;
	oldTime=0;
	timeIndex=0;
	timeModule=0;
	timeSMonth=10;			// April: daylight saving start
	timeSWeek=1;			// First
	timeSDay=6;				// Sunday	
	timeEMonth=4;			// October: daylight savings end
	timeEWeek=1;			// First
	timeEDay=6;				// Sunday 
	f_open(&flog, LOG_FILE, FA_CREATE_ALWAYS | FA_WRITE);
	f_close(&flog);
	// createDynamicVariablesFile();
	// while(1);
	// begin new log...
	logString("System: Master Reset. Ethernet Web Server Version: ", 0);
	ftoa(iBuffer, FIRMWARE_VERSION, 2);
	logString(iBuffer, 0);
	logString("\r\n", 0);
	logString("System: Extracting Settings...\r\n", 0);
	initVariables();
	extractAllSettings();
	// Initialize Stack and application related NV variables into AppConfig.
	InitAppConfig();
  	// Initialize core stack layers (MAC, ARP, TCP, UDP) and
	// application modules (HTTP, SNMP, etc.)

	//	putrsUART("Hello World\r\n");
	//	
	//	while(1)
	//	{
	//	LED0_IO=0;
	//	DelayMs(500);
	//	LED0_IO=1;
	//	DelayMs(500);
	//	}	

	//putrsUART("StackInit\r\n");
    StackInit();
	//putrsUART("Finished StackInit\r\n");
	// Initialize any application-specific modules or functions/
	// For this demo application, this only includes the
	// UART 2 TCP Bridge
	#if defined(STACK_USE_UART2TCP_BRIDGE)
	UART2TCPBridgeInit();
	#endif
	procEmailNum=0;
	emailSendLog=0;
	emailReset=1;
	SMTPResolved=1;
	NTPTimeActive=0;
	lastKnownIP.Val=0;

    #if defined(STACK_USE_ICMP_SERVER)
	lastPing=0;
	lastPingIP.Val=0;
    #endif
    lastFTP =0;
    lastFTPIP.Val=0;
    lastFTPResult=-1;
    lastFileType= HTTP_UNKNOWN;
    while(1)
    {
		if(emailSendLog)
		{
		 if(emailPending==0)
		 {
		 emailPending=1;
		 strcpypgm2ram((char*)emailFile, (char*)"log.txt");
		 if(emailReset)strcpypgm2ram((char*)emailSubject, (char*)"Master Reset"); 
		else 
		{
			DisplayIPValue(iBuffer, AppConfig.MyIPAddr);
			outp=(BYTE*)strcpyram2ram((BYTE*)emailSubject, (BYTE*)"New IP Address: ");
			strcpyram2ram((BYTE*)outp, (BYTE*)iBuffer);
		}
		emailSendLog=0;
		emailReset=0;
		}
		}

		t0=TickGet();
        if((t0 - t1) >= TICK_SECOND/2)
        {
            t1 = t0;
            LED0_IO ^= 1;
        }
		
		if(LED1_IO==0)
		{
		if((t0 - t2) >= TICK_SECOND/5)
		{	
			t2 = t0;
			LED1_IO = 1;
		}
		} else
		{
			t2 = t0;
		}

        // This task performs normal stack task including checking
        // for incoming packet, type of packet and calling
        // appropriate stack entity to process it.
        StackTask();
        
        // This tasks invokes each of the core stack application tasks
        StackApplications();

		// Process application specific tasks here.
		// For this demo app, this will include the Generic TCP 
		// client and servers, and the SNMP, Ping, and SNMP Trap
		// demos.  Following that, we will process any IO from
		// the inputs on the board itself.
		// Any custom modules or processing you need to do should
		// go here.
#if(DEBUG_STACK_POSITION)
		putrsUART("#16.\r\n");
#endif
		#if defined(STACK_USE_GENERIC_TCP_CLIENT_EXAMPLE)
		GenericTCPClient();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#17.\r\n");
#endif	
		#if defined(STACK_USE_GENERIC_TCP_SERVER_EXAMPLE)
		GenericTCPServer();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#18.\r\n");
#endif	
		#if defined(STACK_USE_SMTP_CLIENT)
		SMTPDemo();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#19.\r\n");
#endif
		#if defined(STACK_USE_ICMP_CLIENT)
		PingDemo();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#20.\r\n");
#endif
		#if defined(STACK_USE_SNMP_SERVER) && !defined(SNMP_TRAP_DISABLED)
		SNMPTrapDemo();
		if(gSendTrapFlag)
			SNMPSendTrap();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#21.\r\n");
#endif
		#if defined(STACK_USE_BERKELEY_API)
		BerkeleyTCPClientDemo();
		BerkeleyTCPServerDemo();
		BerkeleyUDPClientDemo();
		#endif

#if(DEBUG_STACK_POSITION)
		putrsUART("#22.\r\n");
#endif

		for(i=0; i<NUM_VARS; i++)
		{
		if(variable[i].logp>0)
		{
			if((t0-variable[i].lastlogp)>= (TICK_SECOND*60*variable[i].logp))
			{
				variable[i].lastlogp=t0;
				getLogEmailFileName(varlogFileName, i);
				if(variable[i].lognum<=1)
				{
    				variable[i].lognum=1;
    				f_unlink(varlogFileName);
                }				
				logStringFile(varlogFileName, variable[i].name, 1);
				logStringFile(varlogFileName, ": ", 0);
				ftoa(valueBuffer, variable[i].value, variable[i].decp);
				logStringFile(varlogFileName, valueBuffer, 0);
				logStringFile(varlogFileName, ".\r\n", 0);
				variable[i].lognum++;
			}
		}
		}
		ProcessIO();
        // If the DHCP lease has changed recently, write the new
        // IP address to the LCD display, UART, and Announce service
		ProcessEmails();

#if(DEBUG_STACK_POSITION)
		putrsUART("#23.\r\n");
#endif

		if(currentIPAddress.Val!=AppConfig.MyIPAddr.Val)
		{
			myDHCPBindCount = DHCPBindCount;
			currentIPAddress.Val=AppConfig.MyIPAddr.Val;
			logString("System: New IP Address: ", 0);
			DisplayIPValue(iBuffer, AppConfig.MyIPAddr);
			logString(iBuffer, 0);
			logString("\r\n", 0);
			#if defined(STACK_USE_ANNOUNCE)
				AnnounceIP();
			#endif
			emailSendLog=1;
		}

		// The Time Module Server
		switch(timeIndex)
		{
				case 0:
					if(timeModule)
					{ 	
						timeIndex++; 
						USTA=0x0400;
					} 
					break;
				case 1:
					time=SNTPGetSeconds();
					if(time!=oldTime)timeIndex++; else timeIndex=0;
					break;
				case 2:
					oldTime=time;
					timeIndex++;
					break;			
				case 3:
					putcUART(0x55);			// for auto baud rate detection!
					timeIndex++;
					convertEpoch2Time(time, &destTime);
					break;
				case 4:
					putcUART(MAGIC_HEADER);
					timeIndex++;
					timePtr=(BYTE*)&destTime;
					timeNum=0;
					break;
				case 40:
					if(timeNum<(sizeof(TIME_T)/2)){ putcUART(*timePtr); timePtr+=2; timeNum++; timeIndex=5; } else timeIndex=0;
					break;
				default:
					timeIndex++;
					break;
		}
		
	}
}

// Writes an IP address to the LCD display and the UART as available
void DisplayIPValue(char* outstr, IP_ADDR IPVal)
{
//	printf("%u.%u.%u.%u", IPVal.v[0], IPVal.v[1], IPVal.v[2], IPVal.v[3]);
	BYTE i;

	for(i = 0; i < sizeof(IP_ADDR); i++)
	{
	    	outstr=(char*)uitoa((WORD)IPVal.v[i], (BYTE*)outstr);
			if(i == sizeof(IP_ADDR)-1)
				break;
			*outstr++='.';
	}
	*outstr='\0';
}

// Processes A/D data from the potentiometer
static void ProcessIO(void)
{
	switch(adcState)
	{
		case 0:
			AD1CHS0=3;
			AD1CON1bits.SAMP=1;
			adcState++;
			adcIndex=0;
			break;

		case 1:
#if(DEBUG_STACK_POSITION)
	putrsUART("#A1.\r\n");
#endif
			adcIndex++;
			if(adcIndex>ADC_INDEX_THRESHOLD)
			{
			AD1CON1bits.SAMP=0;
			adcState++;
			adcIndex=0;
			}
			break;

		case 2:
			adcIndex++;
			if((adcIndex<ADC_CONVERT_THRESHOLD)&&(AD1CON1bits.DONE==0))break;
			adcState++;
			if(adcIndex>=ADC_CONVERT_THRESHOLD)adcState++;
			break;

		case 3:
			variable[0].raw=(float)ADC1BUF0;
			variable[0].value=variable[0].gradient*variable[0].raw+variable[0].yintercept;
			if(noChangeCondition(0))variable[0].condition=updateCondition(0, variable[0].condition);
			adcState++;
			break;

		case 4:
			AD1CHS0=2;
			AD1CON1bits.SAMP=1;
			adcState++;
			adcIndex=0;
			break;
		case 5:
#if(DEBUG_STACK_POSITION)
	putrsUART("#A2.\r\n");
#endif
			adcIndex++;
			if(adcIndex>ADC_INDEX_THRESHOLD)
			{
			AD1CON1bits.SAMP=0;
			adcState++;
			adcIndex=0;
			}
			break;

		case 6:
			adcIndex++;
			if((adcIndex<ADC_CONVERT_THRESHOLD)&&(AD1CON1bits.DONE==0))break;
			adcState++;
			if(adcIndex>=ADC_CONVERT_THRESHOLD)adcState++;
			break;

		case 7:
			variable[1].raw=(float)ADC1BUF0;
			variable[1].value=variable[1].gradient*variable[1].raw+variable[1].yintercept;
			if(noChangeCondition(1))variable[1].condition=updateCondition(1, variable[1].condition);
			adcState++;
			break;

		case 8:
			AD1CHS0=1;
			AD1CON1bits.SAMP=1;
			adcState++;
			adcIndex=0;
			break;

		case 9:
#if(DEBUG_STACK_POSITION)
	putrsUART("#A3.\r\n");
#endif
			adcIndex++;
			if(adcIndex>ADC_INDEX_THRESHOLD)
			{
			AD1CON1bits.SAMP=0;
			adcState++;
			adcIndex=0;
			}
			break;

		case 10:
			adcIndex++;
			if((adcIndex<ADC_CONVERT_THRESHOLD)&&(AD1CON1bits.DONE==0))break;
			adcState++;
			if(adcIndex>=ADC_CONVERT_THRESHOLD)adcState++;
			break;

		case 11:
			variable[2].raw=(float)ADC1BUF0;
			variable[2].value=variable[2].gradient*variable[2].raw+variable[2].yintercept;
			if(noChangeCondition(2))variable[2].condition=updateCondition(2, variable[2].condition);
			adcState++;
			break;

		case 12:
			AD1CHS0=0;
			AD1CON1bits.SAMP=1;
			adcState++;
			adcIndex=0;
			break;

		case 13:
#if(DEBUG_STACK_POSITION)
	putrsUART("#A4.\r\n");
#endif
			adcIndex++;
			if(adcIndex>ADC_INDEX_THRESHOLD)
			{
			AD1CON1bits.SAMP=0;
			adcState++;
			adcIndex=0;
			}
			break;

		case 14:
			adcIndex++;
			if((adcIndex<ADC_CONVERT_THRESHOLD)&&(AD1CON1bits.DONE==0))break;
			adcState++;
			if(adcIndex>=ADC_CONVERT_THRESHOLD)adcState=0;
			break;

		case 15:
			variable[3].raw=(float)ADC1BUF0;
			variable[3].value=variable[3].gradient*variable[3].raw+variable[3].yintercept;
			if(noChangeCondition(3))variable[3].condition=updateCondition(3, variable[3].condition);
			adcState=0;
			break;

		default:
			adcState=0;
			break;
	}
}

/****************************************************************************
  Function:
    static void InitializeBoard(void)

  Description:
    This routine initializes the hardware.  It is a generic initialization
    routine for many of the Microchip development boards, using definitions
    in HardwareProfile.h to determine specific initialization.

  Precondition:
    None

  Parameters:
    None - None

  Returns:
    None

  Remarks:
    None
  ***************************************************************************/
static void InitializeBoard(void)
{	
	// Crank up the core frequency
	PLLFBD = 38;				// Multiply by 40 for 160MHz VCO output (8MHz XT oscillator)
	CLKDIV = 0x0000;			// FRC: divide by 2, PLLPOST: divide by 2, PLLPRE: divide by 2
	// LEDs
	LED0_TRIS = 0;
	LED1_TRIS = 0;
	// Port I/O
	AD1PCFGLbits.PCFG9  = 1;	
	AD1PCFGLbits.PCFG10 = 1;	
	AD1PCFGLbits.PCFG11 = 1;	
	AD1PCFGLbits.PCFG12 = 1;	
	// ADC
//    AD1CHS0 = 0;					
	AD1PCFGLbits.PCFG0 = 0;			// Disable digital input on AN0
	AD1PCFGLbits.PCFG1 = 0;			// Disable digital input on AN1
	AD1PCFGLbits.PCFG2 = 0;			// Disable digital input on AN2
	AD1PCFGLbits.PCFG3 = 0;			// Disable digital input on AN3

	TRISBbits.TRISB11=1;
	TRISBbits.TRISB10=0;
	TRISBbits.TRISB9=0;

	SDCS=1;
	EEPROM_CS_IO=1;
	ENC_CS_IO=1;
	ENC_RST_IO=0;

	EEPROM_CS_TRIS=0;
	SDCS_TRIS=0;
	ENC_CS_TRIS=0;
	ENC_RST_TRIS=0;

	ENC_SDI_TRIS=1;
	ENC_SDO_TRIS=0;
	ENC_SCK_TRIS=0;

	RPOR4=0x0800;		// RP9= SPI Clk 
	RPOR5=0x0007;		// RP10= SPI Data Out
	RPINR20=0x090B;		// RP11= SPI Data In
	// ADC
	AD1CON1 = 0x8000;			// Turn on, 10 bit mode
	AD1CON2 = 0x0000;			// AVdd, AVss
	AD1CON3 = 0x1010;		// 16 Tad auto-sample, Tad = 16*Tcy
	AD1CSSL = 0x000F;			// Scan AN0-AN3
	adcState=0;
	adcIndex=0;
	// UART
	UARTTX_TRIS = 0;
	UARTRX_TRIS = 1;
	UMODE = 0x8010;			// Set UARTEN.  Note: this must be done before setting UTXEN
	USTA  = 0x0400;			// UTXEN set
	
	RPINR19=0x0005;			// assign RP5 pin to U2RX function
	RPOR2=0x0005;			// assign RP4 pin to U2TX function
    baudRate= BAUD_RATE;
    UBRG = ((GetPeripheralClock()/baudRate)/16)-1;
		
#if defined(SPIRAM_CS_TRIS)
	SPIRAMInit();
#endif
#if defined(EEPROM_CS_TRIS)
	XEEInit();
#endif

#if defined(SPIFLASH_CS_TRIS)
	SPIFlashInit();
#endif
	InitSPI(0x0F);
}

/*********************************************************************
 * Function:        void InitAppConfig(void)
 *
 * PreCondition:    MPFSInit() is already called.
 *
 * Input:           None
 *
 * Output:          Write/Read non-volatile config variables.
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            None
 ********************************************************************/
// MAC Address Serialization using a MPLAB PM3 Programmer and 
// Serialized Quick Turn Programming (SQTP). 
// The advantage of using SQTP for programming the MAC Address is it
// allows you to auto-increment the MAC address without recompiling 
// the code for each unit.  To use SQTP, the MAC address must be fixed
// at a specific location in program memory.  Uncomment these two pragmas
// that locate the MAC address at 0x1FFF0.  Syntax below is for MPLAB C 
// Compiler for PIC18 MCUs. Syntax will vary for other compilers.
//#pragma romdata MACROM=0x1FFF0
//#pragma romdata

static void InitAppConfig(void)
{
//	AppConfig.Flags.bIsDHCPEnabled = TRUE;
//	AppConfig.Flags.bInConfigMode  = TRUE;
//	memcpypgm2ram((void*)&AppConfig.MyMACAddr, (ROM void*)SerializedMACAddress, sizeof(AppConfig.MyMACAddr));
//	AppConfig.MyIPAddr.Val = MY_DEFAULT_IP_ADDR_BYTE1 | MY_DEFAULT_IP_ADDR_BYTE2<<8ul | MY_DEFAULT_IP_ADDR_BYTE3<<16ul | MY_DEFAULT_IP_ADDR_BYTE4<<24ul;
//	AppConfig.DefaultIPAddr.Val = AppConfig.MyIPAddr.Val;
//	AppConfig.MyMask.Val = MY_DEFAULT_MASK_BYTE1 | MY_DEFAULT_MASK_BYTE2<<8ul | MY_DEFAULT_MASK_BYTE3<<16ul | MY_DEFAULT_MASK_BYTE4<<24ul;
//	AppConfig.DefaultMask.Val = AppConfig.MyMask.Val;
//	AppConfig.MyGateway.Val = MY_DEFAULT_GATE_BYTE1 | MY_DEFAULT_GATE_BYTE2<<8ul | MY_DEFAULT_GATE_BYTE3<<16ul | MY_DEFAULT_GATE_BYTE4<<24ul;
//	AppConfig.PrimaryDNSServer.Val = MY_DEFAULT_PRIMARY_DNS_BYTE1 | MY_DEFAULT_PRIMARY_DNS_BYTE2<<8ul  | MY_DEFAULT_PRIMARY_DNS_BYTE3<<16ul  | MY_DEFAULT_PRIMARY_DNS_BYTE4<<24ul;
//	AppConfig.SecondaryDNSServer.Val = MY_DEFAULT_SECONDARY_DNS_BYTE1 | MY_DEFAULT_SECONDARY_DNS_BYTE2<<8ul  | MY_DEFAULT_SECONDARY_DNS_BYTE3<<16ul  | MY_DEFAULT_SECONDARY_DNS_BYTE4<<24ul;
//	extractAllSettings();
//	putrsUART("Getting MAC Address from EEPROM...\r\n");
//	putrsUART("MAC Address: ");

	// SNMP Community String configuration
	#if defined(STACK_USE_SNMP_SERVER)
	{
		BYTE i;
		static ROM char * ROM cReadCommunities[] = SNMP_READ_COMMUNITIES;
		static ROM char * ROM cWriteCommunities[] = SNMP_WRITE_COMMUNITIES;
		ROM char * strCommunity;
		
		for(i = 0; i < SNMP_MAX_COMMUNITY_SUPPORT; i++)
		{
			// Get a pointer to the next community string
			strCommunity = cReadCommunities[i];
			if(i >= sizeof(cReadCommunities)/sizeof(cReadCommunities[0]))
				strCommunity = "";

			// Ensure we don't buffer overflow.  If your code gets stuck here, 
			// it means your SNMP_COMMUNITY_MAX_LEN definition in TCPIPConfig.h 
			// is either too small or one of your community string lengths 
			// (SNMP_READ_COMMUNITIES) are too large.  Fix either.
			if(strlenpgm(strCommunity) >= sizeof(AppConfig.readCommunity[0]))
				while(1);
			
			// Copy string into AppConfig
			strcpypgm2ram((char*)AppConfig.readCommunity[i], strCommunity);

			// Get a pointer to the next community string
			strCommunity = cWriteCommunities[i];
			if(i >= sizeof(cWriteCommunities)/sizeof(cWriteCommunities[0]))
				strCommunity = "";

			// Ensure we don't buffer overflow.  If your code gets stuck here, 
			// it means your SNMP_COMMUNITY_MAX_LEN definition in TCPIPConfig.h 
			// is either too small or one of your community string lengths 
			// (SNMP_WRITE_COMMUNITIES) are too large.  Fix either.
			if(strlenpgm(strCommunity) >= sizeof(AppConfig.writeCommunity[0]))
				while(1);

			// Copy string into AppConfig
			strcpypgm2ram((char*)AppConfig.writeCommunity[i], strCommunity);
		}
	}
	#endif

#if(STACK_USE_NBNS)
	// Load the default NetBIOS Host Name
	memcpypgm2ram(AppConfig.NetBIOSName, (ROM void*)MY_DEFAULT_HOST_NAME, 16);
	FormatNetBIOSName(AppConfig.NetBIOSName);
#endif

//	BYTE c;
		
	    // When a record is saved, first byte is written as 0x60 to indicate
	    // that a valid record was saved.  Note that older stack versions
		// used 0x57.  This change has been made to so old EEPROM contents
		// will get overwritten.  The AppConfig() structure has been changed,
		// resulting in parameter misalignment if still using old EEPROM
		// contents.
		//XEEReadArray(0x0000, &c, 1);
/*		
if(c == 0x76u)
		{
		    XEEReadArray(0x001, (BYTE*)&AppConfig, sizeof(AppConfig));
			//logString("Retrieved Settings from EEPROM...\r\n", 0);
		}
	    else
		{
			//putrsUART("Before: \r\n");
			//showEEPROMMemory();
			//showAppConfig();
	        SaveAppConfig();
			logString("System: Saving Settings to EEPROM...\r\n", 0);
			//putrsUART("Saved Settings...\r\n");
			//putrsUART("Now: \r\n");
			//showEEPROMMemory();
		}		
*/
}

/*
void SaveAppConfig(void)
{
	// Ensure adequate space has been reserved in non-volatile storage to 
	// store the entire AppConfig structure.  If you get stuck in this while(1) 
	// trap, it means you have a design time misconfiguration in TCPIPConfig.h.
	// You must increase MPFS_RESERVE_BLOCK to allocate more space.
	//	char c;

    XEEBeginWrite(0x0000);
    XEEWrite(0x76);
	XEEEndWrite();
//	XEEReadArray(0x0000, &c, 1);
//	putrsUART("Now c is: ");
//	disA(c);
//	putrsUART("\r\n");
//	putrsUART("Size of appconfig is : ");
//	disWord(sizeof(AppConfig));
//	putrsUART("\r\n");
	XEEBeginWrite(0x001);
    XEEWriteArray((BYTE*)&AppConfig, sizeof(AppConfig));
	XEEEndWrite();
//	XEEReadArray(0x0000, &c, 1);
//	putrsUART("Now it is c is: ");
//	disA(c);
//	putrsUART("\r\n");
}
*/

